
********************************************************************************
*                                                                              *
* 2 by 4 mixed design: Examples 11 and 12                                      *
*                                                                              *
********************************************************************************

clear
set more off

use mixed2by4data

browse

**********************
* Conventional ANOVA *
**********************

// To conduct a conventional mixed ANOVA, Stata wants the data to be in
// long format:

reshape long row, i(person) j(location)  
rename row retention

// Next we run the mixed ANOVA with retention as dependent variable, sunglasses
// as between-subject factor, and location as within-subject factor:

anova retention sunglasses/person|sunglasses location sunglasses#location, ///
	repeated(location) 

// As explained in the annotated syntax of Examples 8 and 9, Stata, in contrast 
// to SPSS, includes in the ANOVA output the SSmodel. However, this SSmodel is 
// defined differently than in Equation 7. In the Stata output, SSmodel is the 
// sum of SSsunglasses, SSlocation, SSsunglasses*location, and 
// SSperson|sunglasses. To obtain the requires SSmodel (i.e., the variance 
// observed to be related to the manipulation of sunglasses and seating 
// location), the SSperson|sunglasses has to be subtracted from the SSmodel as 
// stored in Stata's memory as "e(mss)". Since SSperson|sunglasses is specified 
// as the second factor in the ANOVA command above, it is stored in Stata's 
// memory as "e(ss_2)". Also, the SSlocation*person|sunglasses in equation 13 is 
// labeled SSresidual in the Stata output table. 

// Let us save first the corrected SSmodel to mata: 

mata
SSmodel = st_numscalar("e(mss)")-st_numscalar("e(ss_2)") 
end


**********************************************************
* Example 11: Negative linear effect of seating location *
* without but not with sunglasses?                       *
**********************************************************

// To conduct contrast analysis on mixed designs we once more use the manovatest
// postestimation command of the MANOVA produre. Since mixed designs involve a
// combination of between- and within-subject factors both a vector L (for the
// between-subject part of the design) and a vector M (for the within-subject 
// part of the design) have to be specified to describe the theorectical
// prediction. The vector L is input for the test option (as in Examples 1 to 
// 7), and the vector M is input for the ytransform option of manovatest (as in 
// Examples 8 to 10). As explained in the annotated syntax of Example 6, Stata,
// in contrast to SPSS, cannot test two separate theorectical predictions 
// in one run. Instead the analysis has to be repeated for each contrast 
// separately. Since the MANOVA procedure requires the data to be in wide
// format, let us re-open the original dataset:

use mixed2by4data, clear
browse

// Next we conduct the MANOVA with the four repeated retentions measurements 
// (variables row1 to row4) as dependent variables, and the "sunglasses" 
// variable as between-subject factor. As in the previous example, the outcome 
// of this MANOVA is of little interest and can be disregarded:

manova row1 row2 row3 row4= sunglasses

// Next we define the two vectors, L and M, with the contrast weights that 
// the first theorectical prediction: a negative linear decrease in retention
// across seating location when the teacher is not wearing sunglasses. When 
// specifying the L matrix keep in mind that Stata requires the weight of the 
// grand mean or intercept (in this case 1) to be included in the vector L (see
// the annotated syntax of Example 1): 

matrix M = (3, 1, -1, -3)
matrix L = (1, 0, 1)

// The vector L is input to the test option and M to the ytransform option of 
// the manovatest postestimation command:

manovatest, test(L) ytransform(M)

// Next we store SScontrast in Mata's memory in exactly the same way as in other
// examples, so that it is available for calculating ETA^2 later on:

mata
SScontrast = st_matrix("r(H)")
end

// Next we can calculate the C-estimate for this first theorectical prediction,
// which requires the cell means (stored in Stata's memory in a vector e(b). Let
// us obtain and display those cell means, and store them in Mata's memory:

mean row1 row2 row3 row4, over(sunglasses)

mata
Means = st_matrix("e(b)") 
Means
end

// The vector Means in which the cell means are stored contains the cell means 
// in the following order: μ11, μ21, μ12, μ22, μ13, μ23, μ14, and μ24. To 
// calculate the C-estimate these cell means have to be multiplied by the 
// contrast weights as defined in W (the product of the transposed L vector and 
// the M vector). This W, however, is a 2 by 4 matrix, and should be
// converted to a row vectors in which the weigths arranged in a similar order
// as means in the vector Means:

mata
M = st_matrix("M")		   // Read vector M into Mata
L = st_matrix("L")	       // Read vector L into Mata
L = L[1,1::(cols(L) - 1)]  // Remove the weight of intercept from vector L
W = L'*M				   // Construct matrix W from L and M
W = vec(W)'				   // Transform W into a row vector in which the order
						   // of the weights match the order of means in Means
Cest = rowsum(W :* Means)  // Calculate the C-estimate (i.e., the weighted sum
						   // of cell means using contrast weights as weights)
Cest					   // Display the C-estimate
end

// Next, we can calculate ETA^2 alerting by dividing SScontrast with SSmodel
// as obtained with the conventional ANOVA. However, since an M matrix is 
// involved, SScontrast needs to be adjusted by dividing it with the sum of 
// Squared M matrix weights:

mata
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting
end

// Finally, we can repeat the same analysis for the second prediction of a
// negative linear effect of seating location on retention when the teacher was
// wearing dark sunglasses. 

matrix M = (3, 1, -1, -3)
matrix L = (0, 1, 1)

manova row1 row2 row3 row4= sunglasses
manovatest, test(L) ytransform(M)

mata
SScontrast = st_matrix("r(H)")
M = st_matrix("M")
L = st_matrix("L")	      
L = L[1,1::(cols(L) - 1)]  
W = vec(L'*M)'				  
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting						   
Cest = rowsum(W :* Means)  
Cest						   
end


*****************************************************
* Example 12: Negative linear trend more pronounced *
* without than with sunglasses?                     *
*****************************************************

// The theorectical prediction in this example is tested in exactly the same
// way as for the predictions in Example 11, except of course for the contrast
// weights being different. Note that the SSmodel, and the observed
// cell means (vector "Means") are already available from the analysis of 
// Example 11:

matrix M = (3, 1, -1, -3)
matrix L = (1, -1, 0)

manova row1 row2 row3 row4= sunglasses
manovatest, test(L) ytransform(M)

mata
SScontrast = st_matrix("r(H)")
M = st_matrix("M")
L = st_matrix("L")	      
L = L[1,1::(cols(L) - 1)]  
W = vec(L'*M)'				  
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting						   
Cest = rowsum(W :* Means)  
Cest						   
end
